﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;

namespace MegaManRipoff.MainGameClasses
{
    /// <summary>
    /// Defines a background for a room.
    /// </summary>
    class Background
    {
        #region Member Variables

        /// <summary>
        /// The current level.
        /// </summary>
        protected Level _level;

        /// <summary>
        /// A collection of rooms that this background belongs to.
        /// </summary>
        protected List<Room> _rooms;

        /// <summary>
        /// The list of background elements and their assets.
        /// </summary>
        List<BackgroundElement> _backgroundElements;

        #endregion

        /// <summary>
        /// Creates a new background.
        /// </summary>
        public Background()
        {
            _backgroundElements = new List<BackgroundElement>();
            _rooms = new List<Room>();
        }

        #region Methods

        /// <summary>
        /// Loads the assets required by this background's elements.
        /// </summary>
        /// <param name="content">The game's content manager.</param>
        /// <param name="contentPath">The directory of the background element textures.</param>
        public void LoadContent(ContentManager content, string textureDirectory)
        {
            foreach (BackgroundElement backgroundElement in _backgroundElements)
            {
                backgroundElement.LoadContent(content, textureDirectory);
            }
        }

        /// <summary>
        /// Assigns a level to this background. This is a poor workaround that deals
        /// with the order that level properties are loaded into the game. Oh well.
        /// </summary>
        /// <param name="level">The level.</param>
        public void AssignLevel(Level level)
        {
            _level = level;
        }

        /// <summary>
        /// Assigns a room to this background.
        /// </summary>
        /// <param name="room">The room.</param>
        public void AssignRoom(Room room)
        {
            _rooms.Add(room);
        }

        /// <summary>
        /// Creates a new background element.
        /// </summary>
        /// <param name="backgroundElement">The new background element and the path of its texture
        /// asset.</param>
        public void AddBackgroundElement(BackgroundElement backgroundElement)
        {
            _backgroundElements.Add(backgroundElement);
        }

        /// <summary>
        /// Updates the background's elements.
        /// </summary>
        /// <param name="gameTime">The current snapshot of time.</param>
        public void Update(GameTime gameTime)
        {
            //Update the list of background elements.
            foreach (BackgroundElement backgroundElement in _backgroundElements)
            {
                backgroundElement.Update(gameTime);
            }
        }

        /// <summary>
        /// Draws the background's elements if necessary.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to draw to.</param>
        public void Draw(SpriteBatch spriteBatch)
        {
            //If one of the rooms that uses this background is active, draw the background.
            foreach (Room ourRoom in _rooms)
            {
                if (_level.ActiveRooms.Contains(ourRoom))
                {
                    //Draw the list of background elements.
                    foreach (BackgroundElement backgroundElement in _backgroundElements)
                    {
                        backgroundElement.Draw(spriteBatch, ourRoom.Area, ourRoom.Camera.Area);
                    }
                }
            }
        }

        #endregion
    }
}
