﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;

namespace MegaManRipoff.MainGameClasses
{
    /// <summary>
    /// The base class of a bullet. Bullets fly around and deal damage to
    /// entities, but can't be damaged themselves.
    /// </summary>
    class Bullet
    {
        #region Member Variables

        /// <summary>
        /// The room the bullet occupies.
        /// </summary>
        protected Room _room;

        /// <summary>
        /// The position of the bullet.
        /// </summary>
        protected Vector2 _position;

        /// <summary>
        /// The speed vector of the bullet.
        /// </summary>
        protected Vector2 _speed;

        /// <summary>
        /// Store a reference to the instance that created this bullet so that other entities can
        /// collide with it accordingly.
        /// </summary>
        protected Entity _bulletOwner;

        /// <summary>
        /// The bullet's animation.
        /// </summary>
        protected Animation _animation;

        /// <summary>
        /// The amount of damage to be dealt to an entity upon collision.
        /// </summary>
        protected int _damage;

        /// <summary>
        /// Should the bullet be destroyed upon a collision?
        /// </summary>
        readonly bool _destroyOnCollision;

        #endregion

        #region Properties

        /// <summary>
        /// Gets the bullet's position.
        /// </summary>
        public Vector2 Position
        {
            get { return _position; }
        }

        /// <summary>
        /// Gets the bullet's hitbox.
        /// </summary>
        public virtual Rectangle Hitbox
        {
            get { return new Rectangle(); }
        }

        /// <summary>
        /// A reference to the instance that created this bullet, so that other entities can
        /// collide with it accordingly.
        /// </summary>
        public Entity BulletOwner
        {
            get { return _bulletOwner; }
        }

        /// <summary>
        /// Accessor for the amount of damage the bullet should do.
        /// </summary>
        public int Damage
        {
            get { return _damage; }
        }

        /// <summary>
        /// Accessor for whether or not the bullet should be destroyed upon a collision.
        /// </summary>
        public virtual bool DestroyOnCollision
        {
            get { return _destroyOnCollision; }
        }

        #endregion

        /// <summary>
        /// Creates a new bullet.
        /// </summary>
        /// <param name="room">The reference to the current room.</param>
        /// <param name="position">The position to create the bullet.</param>
        /// <param name="bulletOwner">The entity that created the bullet.</param>
        /// <param name="damage">How much damage the bullet should deal.</param>
        /// <param name="speed">The bullet's speed vector.</param>
        /// <param name="destroyOnCollision">Whether or not the bullet should be destroyed
        /// when a collision is encountered.</param>
        /// <param name="initialiseAnimations">Whether or not the bullet constructor calls
        /// InitialiseAnimations().</param>
        protected Bullet(Room room, Vector2 position, Entity bulletOwner, int damage,
            Vector2 speed, bool destroyOnCollision, bool initialiseAnimations)
        {
            _room = room;
            _position = position;
            _bulletOwner = bulletOwner;
            _damage = damage;
            _speed = speed;
            _destroyOnCollision = destroyOnCollision;

            //Initialise animations if necessary.
            if (initialiseAnimations)
                InitialiseAnimations();
        }

        #region Methods

        /// <summary>
        /// Initialises the animations for this bullet.
        /// </summary>
        protected virtual void InitialiseAnimations()
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Removes the bullet from the room.
        /// </summary>
        public virtual void Remove()
        {
            _room.Remove(this);
        }

        /// <summary>
        /// Causes the bullet to be deflected (and thus useless).
        /// </summary>
        public virtual void Deflect()
        {
            //Destroy the bullet if there is no overridden method.
            Remove();
        }

        /// <summary>
        /// Moves the bullet according to its speed vector.
        /// </summary>
        /// <param name="gameTime">The current snapshot of time.</param>
        public virtual void Update(GameTime gameTime)
        {
            _position += _speed;
        }

        /// <summary>
        /// Draws the bullet to the screen. This method does nothing and is
        /// designed to be overridden.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to draw to.</param>
        public virtual void Draw(SpriteBatch spriteBatch)
        {
            throw new NotImplementedException();
        }

        #endregion
    }
}
