﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;
using System.Diagnostics;

namespace MegaManRipoff.MainGameClasses
{
    /// <summary>
    /// A big evil killing thing (BEKT) that takes focus of the camera
    /// and slowly moves upwards until it hits the top of the screen.
    /// It will kill the player upon contact. It's also more cute than
    /// it is evil.
    /// </summary>
    class BigEvilKillingThing : Enemy, IFocusable
    {
        #region Member Variables

        /// <summary>
        /// The entity's texture.
        /// </summary>
        static Texture2D _texture;

        /// <summary>
        /// Holds all states that the BEKT can be in.
        /// </summary>
        enum State
        {
            Moving, Stopped
        }

        /// <summary>
        /// The current state of the BEKT.
        /// </summary>
        State _currentState;

        /// <summary>
        /// The enemy's animation.
        /// </summary>
        Dictionary<State, Animation> _animationSet;

        /// <summary>
        /// The maximum amount of health the BEKT can have.
        /// </summary>
        static int _maxHealth;

        /// <summary>
        /// The amount of damage the BEKT deals upon collision.
        /// </summary>
        static int _damage;

        /// <summary>
        /// The maximum velocity of the BEKT.
        /// </summary>
        static float _maxSpeed;

        /// <summary>
        /// The rate of change of velocity of the BEKT.
        /// </summary>
        static float _accel;

        /// <summary>
        /// Sound effect used when the BEKT reaches the end of the screen.
        /// </summary>
        static SoundEffect _thudSound;

        #endregion

        #region Properties

        /// <summary>
        /// Gets the hitbox for this BEKT.
        /// </summary>
        public override Rectangle Hitbox
        {
            get
            {
                if (Game1.DEBUG)
                    return new Rectangle((int)_position.X, (int)_position.Y + 16, 512, 96);
                else
                    return new Rectangle((int)_position.X, (int)_position.Y + 8, 512, 104);
            }
        }

        /// <summary>
        /// Gets the amount of damage BEKTs deal.
        /// </summary>
        public override int Damage
        {
            get { return _damage; }
        }

        /// <summary>
        /// Gets the focus point for this BEKT.
        /// </summary>
        public Vector2 FocusPoint
        {
            get 
            {
                return new Vector2(Hitbox.Center.X,
                                   Hitbox.Bottom - (_room.Camera.Area.Height / 2));
            }
        }

        #endregion

        public BigEvilKillingThing(Room room, Vector2 position)
            : base(room, position)
        {
            _health = _maxHealth;
            _currentState = State.Moving;
        }

        #region Methods

        /// <summary>
        /// Initialises the BEKT's variables, dependent on the difficulty.
        /// </summary>
        /// <param name="gameDifficulty">The game's difficulty setting.</param>
        new public static void Initialise(GameDifficulty gameDifficulty)
        {
            switch (gameDifficulty)
            {
                case GameDifficulty.Easy:
                    _maxSpeed = 0.25f;
                    _accel = -0.0012f;
                    break;
                case GameDifficulty.Hard:
                    _maxSpeed = 0.6f;
                    _accel = -0.009f;
                    break;
                default:
                    _maxSpeed = 0.4f;
                    _accel = -0.003f;
                    break;
            }
            _maxHealth = 10000;
            _damage = 10000;
        }

        /// <summary>
        /// Initialises the animations for this BEKT.
        /// </summary>
        protected override void InitialiseAnimations()
        {
            _animationSet = new Dictionary<State, Animation>();

            //The animations differ in debug mode.
            if (Game1.DEBUG)
            {
                Animation animation = new Animation(new Tuple<Rectangle, Vector2, int>
                    (new Rectangle(0, 0, 512, 96), new Vector2(0, 16), 0), 0);
                _animationSet.Add(State.Moving, animation);
                _animationSet.Add(State.Stopped, animation);
            }
            //In non-debug mode, the animations have more frames because it uses
            //a different texture.
            else
            {
                //Moving.
                Animation moving = new Animation(new Tuple<Rectangle, Vector2, int>
                    (new Rectangle(0, 0, 512, 125), Vector2.Zero, 0), 0);
                _animationSet.Add(State.Moving, moving);

                Animation stopped = new Animation(new Tuple<Rectangle, Vector2, int>
                    (new Rectangle(0, 0, 512, 125), Vector2.Zero, 5), 0);
                stopped.AddFrame(new Tuple<Rectangle, Vector2, int>
                    (new Rectangle(0, 125, 512, 125), Vector2.Zero, 5));
                stopped.AddFrame(new Tuple<Rectangle, Vector2, int>
                    (new Rectangle(0, 250, 512, 125), Vector2.Zero, 0));
                _animationSet.Add(State.Stopped, stopped);
            }
        }

        /// <summary>
        /// Loads the assets required by the BEKT.
        /// </summary>
        /// <param name="content">The game's content manager.</param>
        new public static void LoadContent(ContentManager content)
        {
            //Load a different texture if we're in debug mode. This one's for you, Ross.
            if (Game1.DEBUG)
                _texture = content.Load<Texture2D>("Images\\BigEvilKillingThingDebug");
            else
                _texture = content.Load<Texture2D>("Images\\BigEvilKillingThing");

            _thudSound = content.Load<SoundEffect>("Sounds\\Thud");
        }

        /// <summary>
        /// Applies movement to this BEKT.
        /// </summary>
        private void Movement()
        {
            //If we're not travelling at max speed yet, accelerate.
            if (Math.Abs(_speed.Y) < _maxSpeed)
                _speed.Y = MathHelper.Clamp(_speed.Y + _accel, -_maxSpeed, _maxSpeed);

            //Now apply speed to our position.
            _position += _speed;
        }

        /// <summary>
        /// Checks if this BEKT has reached the end of the room yet. If so,
        /// stops it moving.
        /// </summary>
        private void DetectEndOfRoom()
        {
            //If the bottom of this BEKT is above the point where the bottom
            //of the camera would be if it was at the top of the room, it means
            //that the camera could not move any further and this BEKT would appear
            //to float. SO, reset our position accordingly and stop moving.
            if (Hitbox.Bottom <= (_room.Area.Top + _room.Camera.Area.Height))
            {
                //Stop moving and play a sound effect.
                if (_currentState != State.Stopped)
                {
                    _currentState = State.Stopped;
                    _thudSound.Play();
                    _room.Camera.StartShaking();
                }

                //Change focus back to the player, if he's not dead.
                if (_room.PlayerOne.CurrentState != Player.State.Dying)
                {
                    _room.Camera.Focus = _room.PlayerOne;
                }
            }
        }

        /// <summary>
        /// Keeps the camera at the top of the screen when the BEKT has stopped
        /// moving by changing the focus to this object if the player moves
        /// below the middle of the screen (or dies).
        /// </summary>
        private void KeepCameraAtTop()
        {
            //If the player is alive and above the middle of the top of the room,
            //he can have the focus.
            if (_room.PlayerOne.CurrentState != Player.State.Dying
                && _room.PlayerOne.FocusPoint.Y < _room.Area.Top + (_room.Camera.Area.Height / 2))
            {
                _room.Camera.Focus = _room.PlayerOne;
            }
            //Otherwise, keep this BEKT focussed.
            else
            {
                _room.Camera.Focus = this;
            }

            Debug.WriteLine(_room.Camera.Focus);
        }

        /// <summary>
        /// Updates the BEKT.
        /// </summary>
        /// <param name="gameTime">The current snapshot of time.</param>
        public override void Update(GameTime gameTime)
        {
            //If we're set to move...
            if (_currentState == State.Moving)
            {
                //If this BEKT not the camera's focus, set it to be.
                if (_room.Camera.Focus != this)
                {
                    _room.Camera.Focus = this;
                }

                //Move.
                Movement();

                //Check if we need to stop moving.
                DetectEndOfRoom();
            }
            //If we're not set to move...
            else
            {
                //Keep the camera locked to the top of the screen.
                KeepCameraAtTop();
            }

            //Update animations.
            _animationSet[_currentState].Update();
        }

        /// <summary>
        /// Draws the BEKT to the screen.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to draw to.</param>
        public override void Draw(SpriteBatch spriteBatch)
        {
            //Get the necessary details from the current animation frame.
            Rectangle currentFrameSource = _animationSet[_currentState].CurrentFrame.Item1;
            Vector2 currentFrameOffset = _animationSet[_currentState].CurrentFrame.Item2;

            //Draw the animation frame.
            spriteBatch.Draw(_texture,
                             _position + currentFrameOffset,
                             currentFrameSource,
                             Color.White);
        }

        #endregion
    }
}
