﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Input;
using MegaManRipoff.UI;

namespace MegaManRipoff.MainGameClasses
{
    /// <summary>
    /// A class that creates a Quick Man-esque laser beam after a certain period
    /// of time. Quick Man lasers appeared in one stage in Mega Man 2 - they
    /// are very fast, instant-kill beams that drove players insane in 1990, and
    /// now they'll do the same to you here!!
    /// </summary>
    class QuickManLaserGenerator : Enemy
    {
        #region Member Variables

        /// <summary>
        /// Keeps track of how long to wait before creating the laser, in ticks.
        /// </summary>
        int _delay;

        /// <summary>
        /// The amount of damage Quick Man lasers deal.
        /// </summary>
        static int _laserDamage;

        /// <summary>
        /// The horizontal speed of Quick Man lasers.
        /// </summary>
        static int _laserSpeed;

        /// <summary>
        /// The sound effect instance used when a laser is created.
        /// </summary>
        static SoundEffect _laserSound;

        #endregion

        /// <summary>
        /// Creates a new Quick Man laser generator.
        /// </summary>
        /// <param name="room">The current room.</param>
        /// <param name="position">The position of the laser.</param>
        /// <param name="delay">How long to wait before creating the laser, in ticks.</param>
        /// <param name="direction">The direction of the laser.</param>
        public QuickManLaserGenerator(Room room, Vector2 position, int delay, Direction direction)
            : base(room, position)
        {
            _delay = delay;
            _direction = direction;
        }

        #region Methods

        /// <summary>
        /// Initialises the Quick Man laser generator's static properties according to the current
        /// difficulty setting.
        /// </summary>
        /// <param name="gameDifficulty">The game's current difficulty setting.</param>
        new public static void Initialise(GameDifficulty gameDifficulty)
        {
            switch (gameDifficulty)
            {
                case GameDifficulty.Easy:
                    _laserDamage = 14;
                    _laserSpeed = 5;
                    break;
                case GameDifficulty.Hard:
                    _laserDamage = 28;
                    _laserSpeed = 7;
                    break;
                default:
                    _laserDamage = 28;
                    _laserSpeed = 6;
                    break;
            }
        }

        /// <summary>
        /// This method exists so that a NotImplementedException isn't thrown.
        /// </summary>
        protected override void InitialiseAnimations()
        {
            //Do nothing - the Quick Man laser generator doesn't have animations.
        }

        /// <summary>
        /// Loads the assets required by the Quick Man laser generator and its lasers.
        /// </summary>
        /// <param name="content">The game's content manager.</param>
        new public static void LoadContent(ContentManager content)
        {
            //Load a different sound if on debug mode
            if (Game1.DEBUG)
                _laserSound = content.Load<SoundEffect>("Sounds\\LaserDebug");
            else
                _laserSound = content.Load<SoundEffect>("Sounds\\Laser");

            //Load the laser's content.
            QuickManLaser.LoadContent(content);
        }

        /// <summary>
        /// Updates the Quick Man laser generator.
        /// </summary>
        /// <param name="gameTime">The current snapshot of time.</param>
        public override void Update(GameTime gameTime)
        {
            //Decrement the timer; if it is now zero, create a laser.
            if (--_delay <= 0)
            {
                _room.Add(new QuickManLaser(_room, _position, this,
                    _laserDamage, new Vector2(_laserSpeed * (int)_direction, 0)));

                //Play the sound effect. If L is held, play the sound effect at random pitches
                //as an easter egg.
                if (InputHelper.IsKeyHeld(Keys.L))
                    _laserSound.Play(1, (float)(new Random().NextDouble() * 2) - 1, 0);
                else
                    _laserSound.Play();

                //Remove this generator to stop it creating more lasers.
                _room.Remove(this);
            }
        }

        /// <summary>
        /// This method exists so that a NotImplementedException isn't thrown.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to not draw to.</param>
        public override void Draw(SpriteBatch spriteBatch)
        {
            //Do nothing.
        }

        #endregion
    }
}
