﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;

namespace MegaManRipoff.MainGameClasses
{
    /// <summary>
    /// The base class for all in-game entities. In general, entity refers 
    /// to some sort of object in the game world that can be interacted with.
    /// </summary>
    class Entity
    {
        #region Member Variables

        /// <summary>
        /// The entity's current position vector.
        /// </summary>
        protected Vector2 _position;

        /// <summary>
        /// The entity's current speed vector.
        /// </summary>
        protected Vector2 _speed;

        /// <summary>
        /// Holds the directions the entity can face.
        /// </summary>
        public enum Direction : int
        {
            /// <summary>
            /// No direction.
            /// </summary>
            None = 0,
            /// <summary>
            /// Left. Returns -1 when cast as an int.
            /// </summary>
            Left = -1,
            /// <summary>
            /// Right. Returns 1 when cast as an int.
            /// </summary>
            Right = 1
        }

        /// <summary>
        /// The entity's current direction.
        /// </summary>
        protected Direction _direction;

        /// <summary>
        /// A reference to the current room being played, so that it can perform collision
        /// calculations upon the room's tilesets, enemies, etc...
        /// </summary>
        protected Room _room;

        #endregion

        #region Properties

        /// <summary>
        /// Gets or sets the entity's current position vector.
        /// </summary>
        public Vector2 Position
        {
            get { return _position; }
            set { _position = value; }
        }

        /// <summary>
        /// Gets or sets the entity's current speed vector.
        /// </summary>
        public Vector2 Speed
        {
            get { return _speed; }
            set { _speed = value; }
        }

        /// <summary>
        /// Gets the entity's current direction.
        /// </summary>
        public Direction CurrentDirection
        {
            get { return _direction; }
        }

        /// <summary>
        /// Gets the entity's hitbox.
        /// </summary>
        public virtual Rectangle Hitbox
        {
            get { return new Rectangle(); }
        }

        #endregion

        /// <summary>
        /// Creates an entity at the given position in the given room.
        /// </summary>
        /// <param name="room">The current room.</param>
        /// <param name="position">The position of the entity in the room.</param>
        protected Entity(Room room, Vector2 position)
        {
            _room = room;
            _position = position;

            //Initalise the entity's animations.
            InitialiseAnimations();
        }

        #region Methods

        /// <summary>
        /// Initialise the entity's static properties based on the difficulty setting.
        /// </summary>
        public static void Initialise(GameDifficulty gameDifficulty)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Create all animations associated with this entity.
        /// </summary>
        protected virtual void InitialiseAnimations()
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Load all assests asscociated with this entity.
        /// </summary>
        /// <param name="content">The content manager to load to.</param>
        public static void LoadContent(ContentManager content)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Performs the entity's update game loop. This method does nothing and is
        /// supposed to be overridden. It exists so that the room update loop can
        /// access each entity's update methods.
        /// </summary>
        /// <param name="gameTime">The game's current snapshot of time.</param>
        public virtual void Update(GameTime gameTime)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Draw the entity to the screen.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to draw to.</param>
        public virtual void Draw(SpriteBatch spriteBatch)
        {
            throw new NotImplementedException();
        }

        #endregion
    }
}
