﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;

namespace MegaManRipoff.MainGameClasses
{
    /// <summary>
    /// An helpful item which can be picked up by the player. They can
    /// either be placed around the level or dropped by enemies.
    /// </summary>
    abstract class Item : Entity
    {
        #region Member Variables

        /// <summary>
        /// Determines if the item should flash and eventually
        /// disappear.
        /// </summary>
        protected bool _timeOut;

        /// <summary>
        /// Keeps track of how long the item should exist before
        /// flashing and eventually disappearing.
        /// </summary>
        protected int _timeOutTimer;

        /// <summary>
        /// The length of time, in ticks, for the item to exist if
        /// it's destined to time out.
        /// </summary>
        const int TimeOutDuration = 500;

        /// <summary>
        /// Determines at which point the item will start
        /// flashing before disappearing.
        /// </summary>
        protected const int FlashThreshold = 120;

        #endregion

        /// <summary>
        /// Creates a new item.
        /// </summary>
        /// <param name="room">The current room.</param>
        /// <param name="position">The position to create the item at.</param>
        /// <param name="timeOut">Whether or not the item should flash and eventually
        /// disappear.</param>
        protected Item(Room room, Vector2 position, bool timeOut)
            : base(room, position)
        {
            _timeOut = timeOut;

            //Decide if we need to set the timer.
            if (timeOut)
            {
                _timeOutTimer = TimeOutDuration;
            }
        }

        #region Methods

        /// <summary>
        /// Causes the item to eventually time out and disappear.
        /// </summary>
        private void TimeOut()
        {
            //Decrement the timer; if it is now zero, remove the instance
            //from the room.
            if (--_timeOutTimer <= 0)
            {
                _room.Remove(this);
            }
        }

        /// <summary>
        /// Moves the item according to gravity, and makes it time out if needbe.
        /// </summary>
        /// <param name="gameTime">The current snapshot of time.</param>
        public override void Update(GameTime gameTime)
        {
            //Move according to gravity.
            _speed = EntityHelper.InduceGravity(_speed);
            EntityHelper.VerticalCollision(_room, Hitbox, ref _position, ref _speed);

            //Begin timing out, if needbe.
            if (_timeOut)
            {
                TimeOut();
            }
        }

        /// <summary>
        /// When implemented in an inherting class, draws the item to the screen.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to draw to.</param>
        abstract public override void Draw(SpriteBatch spriteBatch);

        #endregion
    }
}
