﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;

namespace MegaManRipoff.MainGameClasses
{
    /// <summary>
    /// A small health capsule that gives the player a tiny
    /// piece of health. Common as dirt.
    /// </summary>
    class SmallHealth : Item
    {
        #region Member Variables

        /// <summary>
        /// The small health capsule's texture.
        /// </summary>
        static Texture2D _texture;

        /// <summary>
        /// The amount to boost the player's health by.
        /// </summary>
        static int _healthBoost;

        /// <summary>
        /// The health capsule's animation.
        /// </summary>
        Animation _animation;

        #endregion

        #region Properties

        /// <summary>
        /// Gets the large health capsule's hitbox.
        /// </summary>
        public override Rectangle Hitbox
        {
            get { return new Rectangle((int)_position.X, (int)_position.Y, 10, 10); }
        }

        /// <summary>
        /// Gets the amount to boost the player's health by.
        /// </summary>
        public int HealthBoost
        {
            get { return _healthBoost; }
        }

        #endregion

        /// <summary>
        /// Creates a new small health capsule.
        /// </summary>
        /// <param name="room">The current room.</param>
        /// <param name="position">The position to create the item.</param>
        /// <param name="timeOut">Whether or not the item should flash and eventually
        /// disappear.</param>
        /// <param name="speed">The starting speed vector of the item.</param>
        public SmallHealth(Room room, Vector2 position, bool timeOut, Vector2 speed)
            : base(room, position, timeOut)
        {
            _speed = speed;
        }

        #region Methods

        /// <summary>
        /// Initialises the variables for the small health capsule.
        /// </summary>
        /// <param name="gameDifficulty">The game's current difficulty setting.</param>
        new public static void Initialise(GameDifficulty gameDifficulty)
        {
            switch (gameDifficulty)
            {
                case GameDifficulty.Easy:
                    _healthBoost = 3;
                    break;
                case GameDifficulty.Hard:
                    _healthBoost = 2;
                    break;
                default:
                    _healthBoost = 2;
                    break;
            }
        }

        /// <summary>
        /// Initialises the small health capsule's animation.
        /// </summary>
        protected override void InitialiseAnimations()
        {
            _animation = new Animation(new Tuple<Rectangle, Vector2, int>(
                new Rectangle(0, 0, 10, 10), Vector2.Zero, 8), 0);
            _animation.AddFrame(new Tuple<Rectangle, Vector2, int>(
                new Rectangle(10, 0, 10, 10), Vector2.Zero, 8));
        }

        /// <summary>
        /// Loads the asset for the small health capsule.
        /// </summary>
        /// <param name="content">The game's content manager.</param>
        new public static void LoadContent(ContentManager content)
        {
            _texture = content.Load<Texture2D>("Images\\SmallHealth");
        }

        /// <summary>
        /// Updates the item.
        /// </summary>
        /// <param name="gameTime">The current snapshot of time.</param>
        public override void Update(GameTime gameTime)
        {
            //Update animation.
            _animation.Update();

            base.Update(gameTime);
        }

        /// <summary>
        /// Draws the small health capsule.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to draw to.</param>
        public override void Draw(SpriteBatch spriteBatch)
        {
            //Draw the item if we're not flashing.
            if (!_timeOut || (_timeOut && (_timeOutTimer / 2) % 2 == 0
                || _timeOutTimer > FlashThreshold))
            {
                //Get the necessary details from the current animation frame.
                Rectangle currentFrameSource = _animation.CurrentFrame.Item1;
                Vector2 currentFrameOffset = _animation.CurrentFrame.Item2;

                spriteBatch.Draw(_texture,
                                    _position + currentFrameOffset,
                                    currentFrameSource,
                                    Color.White,
                                    0,
                                    Vector2.Zero,
                                    1,
                                    SpriteEffects.None,
                                    0.15f);
            }
        }

        #endregion
    }
}

