﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;

namespace MegaManRipoff.MainGameClasses
{
    /// <summary>
    /// Represents a health bar. Good for displaying health.
    /// </summary>
    class HealthBar : IHudComponent
    {
        #region Member Variables

        /// <summary>
        /// The entity to display the health of.
        /// </summary>
        HealthyEntity _healthyEntity;

        /// <summary>
        /// The maximum value that can be displayed.
        /// </summary>
        readonly int _maxValue;

        /// <summary>
        /// The position of the health bar on the HUD.
        /// </summary>
        Vector2 _position;

        /// <summary>
        /// The colour of the health bar.
        /// </summary>
        Color _colour;

        /// <summary>
        /// The design of the health bar.
        /// </summary>
        public enum Design
        {
            Normal, Boss
        }

        /// <summary>
        /// The current design of this health bar.
        /// </summary>
        Design _design;

        /// <summary>
        /// The texture of the normal health bar.
        /// </summary>
        static Texture2D _normalTexture;

        /// <summary>
        /// The texture of the boss health bar.
        /// </summary>
        static Texture2D _bossTexture;

        /// <summary>
        /// Area on the texture that consists of the top part of the
        /// healthbar.
        /// </summary>
        static Rectangle _topSection;

        /// <summary>
        /// Area on the texture that consists of the middle part of the
        /// healthbar, which will be stretched.
        /// </summary>
        static Rectangle _middleSection;

        /// <summary>
        /// Area on the texture that consists of the bottom part of the
        /// healthbar.
        /// </summary>
        static Rectangle _bottomSection;

        /// <summary>
        /// Area on the texture that consists of the graphic used for
        /// each unit of health.
        /// </summary>
        static Rectangle _unitSection;

        #endregion

        #region Properties

        /// <summary>
        /// Gets the position of the health bar.
        /// </summary>
        public Vector2 Position
        {
            get { return _position; }
        }

        /// <summary>
        /// Gets the entity that the health bar is displaying.
        /// </summary>
        public HealthyEntity HealthyEntity
        {
            get { return _healthyEntity; }
        }

        #endregion

        /// <summary>
        /// Creates a new health bar on the HUD.
        /// </summary>
        /// <param name="healthyEntity">The entity to display the health of.</param>
        /// <param name="position">The position of the health bar on the HUD.</param>
        /// <param name="colour">The colour of the health bar.</param>
        /// <param name="type">The design of the health bar.</param>
        public HealthBar(HealthyEntity healthyEntity, Vector2 position, Color colour, Design design)
        {
            _healthyEntity = healthyEntity;
            _position = position;
            _colour = colour;
            _design = design;

            //Store the maximum value now incase the entity becomes null.
            _maxValue = healthyEntity.MaxHealth;
        }

        #region Methods

        /// <summary>
        /// Loads the assests associated with this component.
        /// </summary>
        /// <param name="content">The game's content manager.</param>
        public static void LoadContent(ContentManager content)
        {
            _normalTexture = content.Load<Texture2D>("Images\\HealthBar");
            _bossTexture = content.Load<Texture2D>("Images\\HealthBarBoss");

            //Define where the sections of the health bar are on the texture.
            _topSection = new Rectangle(0, 0, 13, 7);
            _middleSection = new Rectangle(1, 7, 11, 2);
            _bottomSection = new Rectangle(0, 9, 13, 16);
            _unitSection = new Rectangle(3, 25, 7, 2);
        }

        /// <summary>
        /// Draws the health bar to the screen.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to draw to.</param>
        public void Draw(SpriteBatch spriteBatch)
        {
            //Decide which texture to use.
            Texture2D texture;
            if (_design == Design.Normal)
                texture = _normalTexture;
            else
                texture = _bossTexture;

            //Draw the top part.
            spriteBatch.Draw(texture,
                             new Vector2(_topSection.Location.X,
                                         _topSection.Location.Y) + _position,
                             _topSection,
                             Color.White);

            //Draw the stretched middle part.
            spriteBatch.Draw(texture,
                             new Vector2(_middleSection.Location.X,
                                         _middleSection.Location.Y) + _position,
                             _middleSection,
                             Color.White,
                             0,
                             Vector2.Zero,
                             new Vector2(1, _maxValue),
                             SpriteEffects.None,
                             0.1f);

            //Draw the bottom part.
            spriteBatch.Draw(texture,
                             new Vector2(_bottomSection.Location.X,
                                         (_middleSection.Height * (_maxValue - 1)) + _bottomSection.Location.Y) + _position,
                             _bottomSection,
                             Color.White);

            //Draw each unit of health if the entity is not null.
            if (_healthyEntity != null)
            {
                for (int i = 0; i < _healthyEntity.Health; i++)
                {
                    spriteBatch.Draw(texture,
                                     new Vector2(_unitSection.Location.X,
                                                 _middleSection.Location.Y + ((_maxValue - _healthyEntity.Health + i) * _unitSection.Height)) + _position,
                                     _unitSection,
                                     _colour);
                }
            }
        }

        #endregion
    }
}
