﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;

namespace MegaManRipoff.MainGameClasses
{
    /// <summary>
    /// An eggshell that falls off the screen.
    /// </summary>
    class Eggshell : SpecialEffect
    {
        #region Member Variables

        /// <summary>
        /// The texture for eggshells.
        /// </summary>
        static Texture2D _texture;

        /// <summary>
        /// The eggshell's speed vector.
        /// </summary>
        Vector2 _speed;

        /// <summary>
        /// Represents two halves of an eggshell.
        /// </summary>
        public enum Half
        {
            Left, Right
        }

        /// <summary>
        /// Determines which half of the eggshell this instance is.
        /// </summary>
        Half _currentHalf;

        /// <summary>
        /// Animations for the left and right halves of the eggshell. (The base class
        /// calls InitialiseAnimations before the half value passed in the constructor
        /// can be evaluated).
        /// </summary>
        Animation _animationLeft, _animationRight;

        #endregion

        public Eggshell(Room room, Vector2 position, Half half)
            : base(room, position, false)
        {
            _currentHalf = half;

            //Set the speed vector and animation according to which side of the
            //eggshell this is.
            if (_currentHalf == Half.Left)
            {
                _speed = new Vector2(-3, -5);
                _animation = _animationLeft;
            }
            else
            {
                _speed = new Vector2(3, -5);
                _animation = _animationRight;
            }
        }

        #region Methods

        /// <summary>
        /// Initialises the eggshell's animations.
        /// </summary>
        protected override void InitialiseAnimations()
        {
            //Left eggshell.
            _animationLeft = new Animation(new Tuple<Rectangle, Vector2, int>(
                    new Rectangle(0, 0, 15, 31), Vector2.Zero, 2), 0);
            _animationLeft.AddFrame(new Tuple<Rectangle, Vector2, int>(
                new Rectangle(), Vector2.Zero, 2));

            //Right eggshell.
            _animationRight = new Animation(new Tuple<Rectangle, Vector2, int>(
                    new Rectangle(16, 0, 16, 31), Vector2.Zero, 2), 0);
            _animationRight.AddFrame(new Tuple<Rectangle, Vector2, int>(
                new Rectangle(), Vector2.Zero, 2));
        }

        /// <summary>
        /// Loads the assets required by eggshells.
        /// </summary>
        /// <param name="content">The game's content manager.</param>
        public static void LoadContent(ContentManager content)
        {
            _texture = content.Load<Texture2D>("Images\\Eggshell");
        }

        /// <summary>
        /// Updates the eggshell.
        /// </summary>
        /// <param name="gameTime">The current snapshot of time.</param>
        public override void Update(GameTime gameTime)
        {
            //Move according to gravity.
            _speed = EntityHelper.InduceGravity(_speed);
            _position += _speed;

            //Remove this instance if we move out of the camera view.
            if (!_room.Camera.IsInView(new Rectangle((int)_position.X, (int)_position.Y, 15, 31)))
            {
                _room.Remove(this);
            }

            base.Update(gameTime);
        }

        /// <summary>
        /// Draws the eggshell to the screen.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to draw to.</param>
        public override void Draw(SpriteBatch spriteBatch)
        {
            //Get the necessary details from the current animation frame.
            Rectangle currentFrameSource = _animation.CurrentFrame.Item1;
            Vector2 currentFrameOffset = _animation.CurrentFrame.Item2;

            spriteBatch.Draw(_texture,
                             _position + currentFrameOffset,
                             currentFrameSource,
                             Color.White);
        }

        #endregion
    }
}
