﻿/*using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;

namespace MegaManRipoff
{
    /// <summary>
    /// Dims the screen and creates rain.
    /// </summary>
    class RainCreator : Entity, IEffect
    {
        #region Member Variables

        /// <summary>
        /// The current state of this effect.
        /// </summary>
        EffectState _currentState;

        /// <summary>
        /// The previous state of this effect. This variable is used 
        /// so that the SoundEffectInstance has a chance to
        /// instantiate before its default variables are set.
        /// </summary>
        EffectState _previousState;

        /// <summary>
        /// The current alpha value of the dimming colour.
        /// </summary>
        float _currentAlpha;

        /// <summary>
        /// The maximum alpha value of the dimming colour. Used
        /// for fading in/out.
        /// </summary>
        public const float MaximumAlpha = 0.3f;

        /// <summary>
        /// Dictates how often raindrop objects should be created.
        /// </summary>
        const int RainRate = 2;

        /// <summary>
        /// The direction that the created raindrops move in.
        /// </summary>
        const int RainDirection = 100;

        /// <summary>
        /// The random amount of variation from the set direction that
        /// each raindrop is created at.
        /// </summary>
        const float RainDispersion = 3;

        /// <summary>
        /// The speed that the created raindrops move at.
        /// </summary>
        const float RainSpeed = 20;

        /// <summary>
        /// The list of currently active raindrops.
        /// </summary>
        List<RainDrop> _rainDrops;

        /// <summary>
        /// A list of entities to remove references to later. This list exists to
        /// avoid modifying the collections during a foreach loop.
        /// </summary>
        List<RainDrop> _cleanUpList;

        /// <summary>
        /// Keeps track of how long the effect has faded for.
        /// </summary>
        int _fadeTimer;

        /// <summary>
        /// How long it takes for the rain creator to fade in/out,
        /// in ticks.
        /// </summary>
        const int FadeDuration = 180;

        /// <summary>
        /// Sound effect for the rain.
        /// </summary>
        static SoundEffect _rain;

        /// <summary>
        /// Allows the rain sound effect to loop.
        /// </summary>
        static SoundEffectInstance _rainInstance;

        /// <summary>
        /// The maximum volume value of the rain sound effect instance.
        /// Used for fading in/out.
        /// </summary>
        const float MaximumVolume = 0.5f;

        /// <summary>
        /// The instance's random number generator.
        /// </summary>
        Random _random;

        #endregion

        #region Properties

        /// <summary>
        /// The current state of the effect.
        /// </summary>
        public EffectState CurrentState
        {
            get { return _currentState; }
        }

        /// <summary>
        /// The current alpha value of the dimming colour.
        /// </summary>
        public float CurrentAlpha
        {
            get { return _currentAlpha; }
        }

        #endregion

        public RainCreator(Room room, EffectState state)
            : base(room, Vector2.Zero)
        {
            _previousState = EffectState.Inactive;
            _currentState = state;

            //Initalise the raindrop lists.
            _rainDrops = new List<RainDrop>();
            _cleanUpList = new List<RainDrop>();

            //Initialise the random number generator.
            _random = new Random();
        }

        #region Methods

        /// <summary>
        /// Loads the assets associated with this effect.
        /// </summary>
        /// <param name="content">The game's content manager.</param>
        new public static void LoadContent(ContentManager content)
        {
            _rain = content.Load<SoundEffect>("Sounds\\Listen");

            //Create the rain sound instance, and set it to loop.
            _rainInstance = _rain.CreateInstance();
            _rainInstance.IsLooped = true;

            //Also call the raindrop's LoadContent() method.
            RainDrop.LoadContent(content);
        }

        /// <summary>
        /// Causes the rain effect to fade in.
        /// </summary>
        public void Activate()
        {
            //Set the new state.
            _currentState = EffectState.Activating;

            //Initalise the raindrop list.
            _rainDrops = new List<RainDrop>();
        }

        /// <summary>
        /// Causes the rain effect to fade out.
        /// </summary>
        public void Deactivate()
        {
            //Set the new state.
            _currentState = EffectState.Deactivating;
        }

        /// <summary>
        /// Causes the effect to fade in/out, dependent on the current state.
        /// </summary>
        private void Fade()
        {
            //Make a positive/negative value based on if we're fading in/out.
            int fadeInteger = 1;
            if (_currentState == EffectState.Deactivating) { fadeInteger = -1; }

            //Increase/decrease the volume and alpha value of the dimming rectangle.
            _rainInstance.Volume += (MaximumVolume / (float)FadeDuration) * fadeInteger;
            _currentAlpha += (MaximumAlpha / (float)FadeDuration) * fadeInteger;

            //Decrement the fade timer; if it is now zero, set our state to
            //either active or inactive.
            if (--_fadeTimer <= 0)
            {
                if (_currentState == EffectState.Deactivating)
                {
                    _currentState = EffectState.Inactive;
                }
                else
                {
                    _currentState = EffectState.Active;
                }
            }
        }

        /// <summary>
        /// Sets the alpha and volume variables depending on the current state.
        /// </summary>
        private void SetupState()
        {
            switch (_currentState)
            {
                case EffectState.Inactive:
                case EffectState.Activating:
                    _currentAlpha = 0;
                    _rainInstance.Volume = 0;
                    _fadeTimer = FadeDuration;
                    break;
                case EffectState.Active:
                case EffectState.Deactivating:
                    _currentAlpha = MaximumAlpha;
                    _rainInstance.Volume = MaximumVolume;
                    _fadeTimer = FadeDuration;
                    break;
            }
        }

        /// <summary>
        /// Randomly adds rain to the rain list.
        /// </summary>
        private void CreateRain()
        {
            //Create a random number; if it is equal to zero, then create a
            //raindrop (thus, the probability of creating rain is 1 / RainRate).
            if (_random.Next(0, RainRate) == 0)
            {
                //Create the random x-position.
                float newX = _random.Next(_room.CameraArea.Location.X - 48, 
                                         _room.CameraArea.Right + 48);

                //Create the random direction dispersion.
                float newDispersion = (float)(_random.NextDouble() * RainDispersion * 2) - RainDispersion;

                //Create the instance.
                _rainDrops.Add(new RainDrop(_room,
                                            new Vector2(newX, _room.CameraArea.Top - 48),
                                            this,
                                            MathHelper.ToRadians(RainDirection + newDispersion),
                                            RainSpeed));
            }
        }

        /// <summary>
        /// Plays the rain sound.
        /// </summary>
        private void PlaySound()
        {
            if (_rainInstance.State == SoundState.Stopped)
            {
                _rainInstance.Play();
            }
        }

        /// <summary>
        /// Removes the given raindrop from the list of raindrops.
        /// </summary>
        /// <param name="raindrop">The raindrops to remove.</param>
        public void RemoveRain(RainDrop raindrop)
        {
            _cleanUpList.Add(raindrop);
        }

        /// <summary>
        /// Cleans up all entities that are designated to be removed.
        /// </summary>
        private void CleanUp()
        {
            //Only bother doing something if the list is not empty.
            if (_cleanUpList.Count != 0)
            {
                //Iterate through the list and remove the entities from their
                //designated lists.
                foreach (RainDrop raindrop in _cleanUpList)
                {
                    _rainDrops.Remove(raindrop);
                }

                //Now empty the clean up list.
                _cleanUpList.Clear();
            }
        }

        /// <summary>
        /// Performs the effect's update game loop.
        /// </summary>
        /// <param name="gameTime">The current snapshot of time.</param>
        public override void Update(GameTime gameTime)
        {
            //Set the starting alpha and volume values when the state changes.
            if (_previousState != _currentState)
            {
                SetupState();
                _previousState = _currentState;
            }

            //Only update if we're active.
            if (_currentState != EffectState.Inactive)
            {
                //If we're in the middle of activating or deactivating,
                //perform the fade routine.
                if (_currentState == EffectState.Activating
                    || _currentState == EffectState.Deactivating)
                {
                    Fade();
                }

                //Only create rain if the game is not lagging.
                if (!gameTime.IsRunningSlowly)
                {
                    CreateRain();
                }

                //Update raindrops.
                foreach (RainDrop raindrop in _rainDrops)
                {
                    raindrop.Update(gameTime);
                }

                //Destroy any raindrops that are supposed to be destroyed.
                CleanUp();

                //Play the sound.
                PlaySound();
            }
        }

        /// <summary>
        /// Draws the dimming rectangle and each of the raindrops.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to draw to.</param>
        public override void Draw(SpriteBatch spriteBatch)
        {
            //Temporarily change the blend state for the raindrops.
            BlendState originalBlendState = spriteBatch.GraphicsDevice.BlendState;
            spriteBatch.GraphicsDevice.BlendState = BlendState.NonPremultiplied;

            //Draw the raindrops.
            foreach (RainDrop raindrop in _rainDrops)
            {
                raindrop.Draw(spriteBatch);
            }

            //Now change the blend state back.
            spriteBatch.GraphicsDevice.BlendState = originalBlendState;

            //Create a new, blank texture for the dimming colour and draw it over the room.
            Texture2D texture = new Texture2D(spriteBatch.GraphicsDevice, 1, 1);
            texture.SetData(new Color[] {new Color(0, 0, 0, _currentAlpha)});
            spriteBatch.Draw(texture, _room.Area, Color.White);
        }

        #endregion
    }
}*/
