﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;

namespace MegaManRipoff.MainGameClasses
{
    /// <summary>
    /// A special effect of an explosion used when the player explodes.
    /// </summary>
    class PlayerDeathEffect : SpecialEffect
    {
        #region Member Variables

        /// <summary>
        /// The explosion's texture.
        /// </summary>
        static Texture2D _texture;

        /// <summary>
        /// The explosion's speed vector.
        /// </summary>
        Vector2 _speed;

        #endregion

        /// <summary>
        /// Creates a new explosion effect.
        /// </summary>
        /// <param name="room">The current room.</param>
        /// <param name="position">The position of the new explosion effect.</param>
        /// <param name="speed">The movement speed of the explosion effect.</param>
        public PlayerDeathEffect(Room room, Vector2 position, Vector2 speed)
            : base(room, position, false)
        {
            _speed = speed;
        }

        #region Methods

        /// <summary>
        /// Creates the animation for this explosion special effect.
        /// </summary>
        protected override void InitialiseAnimations()
        {
            //The only animation.
            _animation = new Animation(new Tuple<Rectangle, Vector2, int>
                (new Rectangle(0, 4, 24, 24), new Vector2(-12, -12), 4), 0);
            _animation.AddFrame(new Tuple<Rectangle, Vector2, int>
                (new Rectangle(24, 2, 28, 28), new Vector2(-14, -14), 4));
            _animation.AddFrame(new Tuple<Rectangle, Vector2, int>
                (new Rectangle(52, 1, 30, 30), new Vector2(-15, -15), 4));
            _animation.AddFrame(new Tuple<Rectangle, Vector2, int>
                (new Rectangle(82, 0, 32, 32), new Vector2(-16, -16), 4));
            _animation.AddFrame(new Tuple<Rectangle, Vector2, int>
                (new Rectangle(114, 1, 30, 30), new Vector2(-15, -15), 4));
            _animation.AddFrame(new Tuple<Rectangle, Vector2, int>
                (new Rectangle(144, 0, 32, 32), new Vector2(-16, -16), 4));

            base.InitialiseAnimations();
        }

        /// <summary>
        /// Loads the assets required for the explosion effect.
        /// </summary>
        /// <param name="content">The game's content manager.</param>
        public static void LoadContent(ContentManager content)
        {
            _texture = content.Load<Texture2D>("Images\\Explosion");
        }

        /// <summary>
        /// Moves the effect according to its speed, updates the animations, and
        /// removes the special effect if it leaves the camera view.
        /// </summary>
        /// <param name="gameTime">The current snapshot of time.</param>
        public override void Update(GameTime gameTime)
        {
            _position += _speed;

            //If we leave the camera view, remove this effect from the room.
            if (!_room.Camera.IsInView(new Rectangle((int)_position.X - 15,
                                                     (int)_position.Y - 14,
                                                     31, 28)))
            {
                _room.Remove(this);
            }

            base.Update(gameTime);
        }

        /// <summary>
        /// Draws the explosion effect to the screen.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to draw to.</param>
        public override void Draw(SpriteBatch spriteBatch)
        {
            //Get the necessary details from the current animation frame.
            Rectangle currentFrameSource = _animation.CurrentFrame.Item1;
            Vector2 currentFrameOffset = _animation.CurrentFrame.Item2;

            spriteBatch.Draw(_texture,
                                _position + new Vector2(currentFrameOffset.X, currentFrameOffset.Y),
                                currentFrameSource,
                                Color.White);
        }

        #endregion
    }
}
