﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using System.Diagnostics;

namespace MegaManRipoff.MainGameClasses
{
    /// <summary>
    /// Defines a special effect that does not interact
    /// with any other in-game entity, and only really exists
    /// to make the game prettier. Most effects play an animation
    /// and/or sound before disappearing.
    /// </summary>
    class SpecialEffect
    {
        #region Member Variables

        /// <summary>
        /// The current room.
        /// </summary>
        protected Room _room;

        /// <summary>
        /// The position of this effect.
        /// </summary>
        protected Vector2 _position;

        /// <summary>
        /// The animation frames of this effect.
        /// </summary>
        protected Animation _animation;

        /// <summary>
        /// Whether or not this special effect should be destroyed when its
        /// animation ends.
        /// </summary>
        bool _destroyOnAnimationEnd;

        #endregion

        /// <summary>
        /// Creates a new special effect.
        /// </summary>
        /// <param name="room">The current room.</param>
        /// <param name="position">The position of the new special effect.</param>
        /// <param name="destroyOnAnimationEnd">Whether or not this special effect should
        /// be destroyed when its animation ends.</param>
        protected SpecialEffect(Room room, Vector2 position, bool destroyOnAnimationEnd)
        {
            _room = room;
            _position = position;
            _destroyOnAnimationEnd = destroyOnAnimationEnd;

            InitialiseAnimations();
        }

        #region Methods

        /// <summary>
        /// Subscribes the special effect to the OnAnimationEnd event, if necessary.
        /// </summary>
        protected virtual void InitialiseAnimations()
        {
            //Animation frames should go in the overridden method.

            //Add a listener to the animation's event if needbe.
            if (_destroyOnAnimationEnd)
            {
                _animation.OnAnimationEnd += new AnimationEnd(_animation_OnAnimationEnd);
            }
        }

        /// <summary>
        /// Updates the special effect's animation, if it has one.
        /// </summary>
        /// <param name="gameTime">The current snapshot of time.</param>
        public virtual void Update(GameTime gameTime)
        {
            if (_animation != null)
                _animation.Update();
        }

        /// <summary>
        /// Draws the special effect. This method is designed to be overriden.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to draw to.</param>
        public virtual void Draw(SpriteBatch spriteBatch)
        {
            throw new NotImplementedException();
        }

        #endregion

        #region Event Handles

        /// <summary>
        /// Method called when this special effect's animation ends.
        /// </summary>
        /// <param name="sender">The animation that ended.</param>
        /// <param name="e">Event arugments (none).</param>
        protected void _animation_OnAnimationEnd(object sender, EventArgs e)
        {
            //If the animation that ended is our animation, destroy this special effect.
            if ((Animation)sender == _animation)
                _room.Remove(this);
        }

        #endregion
    }
}
