﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;
using MegaManRipoff.UI;

namespace MegaManRipoff.TitleScreenClasses
{
    /// <summary>
    /// Handles the display of the credits window.
    /// </summary>
    class CreditsWindow
    {
        #region Member Variables

        /// <summary>
        /// The position of the controls window.
        /// </summary>
        Vector2 _position;

        /// <summary>
        /// The texture used for the controls window.
        /// </summary>
        static Texture2D _texture;

        /// <summary>
        /// The font used for the controls window heading.
        /// </summary>
        static SpriteFont _font;

        static readonly string _headerText = "Credits";

        /// <summary>
        /// The states of the controls window.
        /// </summary>
        public enum State
        {
            Closed, Open
        }

        /// <summary>
        /// The current state of the control window.
        /// </summary>
        State _currentState;

        /// <summary>
        /// The sound effect used when the controls window is closed.
        /// </summary>
        static SoundEffect _closedSound;

        #endregion

        #region Properties

        /// <summary>
        /// Gets or sets the current state of the controls window.
        /// </summary>
        public State CurrentState
        {
            get { return _currentState; }
            set { _currentState = value; }
        }

        #endregion

        /// <summary>
        /// Creates a new controls window.
        /// </summary>
        /// <param name="position">The position of the controls window.</param>
        public CreditsWindow()
        {
            _position = new Vector2(64, Camera.VIEWPORT_HEIGHT + 64);
        }

        #region Methods

        /// <summary>
        /// Loads the assets required by the controls window.
        /// </summary>
        /// <param name="content">The game's content manager.</param>
        public static void LoadContent(ContentManager content)
        {
            _texture = content.Load<Texture2D>("Images\\Credits");
            _font = content.Load<SpriteFont>("Fonts\\SegoeUI");
            _closedSound = content.Load<SoundEffect>("Sounds\\Selected");
        }

        /// <summary>
        /// Responds to the user's input.
        /// </summary>
        private void HandleInput()
        {
            //If the select, back or start button is input, close this window.
            if (InputHelper.IsJumpPressed() || InputHelper.IsStartPressed()
                || InputHelper.IsShootPressed())
            {
                _currentState = State.Closed;
                _closedSound.Play();
            }
        }

        /// <summary>
        /// Moves the controls window to its proper position, depending on
        /// its current state.
        /// </summary>
        private void MoveToPosition()
        {
            switch (_currentState)
            {
                case State.Open:
                    _position.Y = MathHelper.SmoothStep(_position.Y, 32, 0.3f);
                    break;
                case State.Closed:
                    _position.Y = MathHelper.SmoothStep(_position.Y, Camera.VIEWPORT_HEIGHT + 64, 0.3f);
                    break;
            }
        }

        /// <summary>
        /// Updates the controls window.
        /// </summary>
        /// <param name="gameTime">The current snapshot of time.</param>
        public void Update(GameTime gameTime)
        {
            //If the window is open, accept user input.
            if (_currentState == State.Open)
                HandleInput();

            MoveToPosition();
        }

        /// <summary>
        /// Draws the controls window to the screen.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to draw to.</param>
        public void Draw(SpriteBatch spriteBatch)
        {
            //Draw the controls window.
            spriteBatch.Draw(_texture, _position, Color.White);

            //Draw the header.
            spriteBatch.DrawString(_font, _headerText, new Vector2(_position.X + 193 - (_font.MeasureString(_headerText).X / 2),
                                                                  _position.Y + 7), Color.Black);
            spriteBatch.DrawString(_font, _headerText, new Vector2(_position.X + 192 - (_font.MeasureString(_headerText).X / 2),
                                                                  _position.Y + 6), Color.White);
        }

        #endregion
    }
}
