﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Security.Permissions;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Windows.Forms.VisualStyles;
using TrackmaniaServerTool.Properties;

namespace TrackmaniaServerTool
{
	public partial class MainForm : Form
	{
		string MapsRootPath = "";
		Tracklist lastSavedTracklist = new Tracklist();

		public MainForm()
		{
			InitializeComponent();

			if (Properties.Settings.Default.ServerExecutableFileName.Length == 0)
			{
				ShowLocateServerExecutableDialog();
			}
			else 
			{
				OnServerExecutableFileNameChanged();
			}

			if (Properties.Settings.Default.TrackListFileName.Length != 0)
			{
				OpenTracklist(Properties.Settings.Default.TrackListFileName);
			}
		}

		#region File Dialog Events
		private void newTracklistToolStripMenuItem_Click(object sender, EventArgs e)
		{
			if (HandleDirtyTracklistBeforeOverwriting())
			{
				lastSavedTracklist = new Tracklist();
				mapListBox.Items.Clear();
				UpdateStartIndexMaximum();
			}
		}

		private void openTracklistToolStripMenuItem_Click(object sender, EventArgs e)
		{
			if (HandleDirtyTracklistBeforeOverwriting())
			{
				ShowOpenTracklistDialog();
			}
		}

		private void saveTracklistToolStripMenuItem_Click(object sender, EventArgs e)
		{
			if (Properties.Settings.Default.TrackListFileName.Length == 0)
			{
				ShowSaveTracklistDialog();
			}
			else
			{
				SaveTracklist(Properties.Settings.Default.TrackListFileName);
			}
		}

		private void saveTracklistAsToolStripMenuItem_Click(object sender, EventArgs e)
		{
			ShowSaveTracklistDialog();
		}

		private void exitToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Close();
		}
		#endregion

		Tracklist GetCurrentTracklist()
		{
			return new Tracklist(mapListBox.Items.Cast<string>().ToArray(), (int)startIndexUpDown.Value);
		}

		bool IsTracklistDirty()
		{
			Tracklist currentTracklist = GetCurrentTracklist();
			return !currentTracklist.Equals(lastSavedTracklist);
		}

		bool HandleDirtyTracklistBeforeOverwriting()
		{
			if (!IsTracklistDirty())
				return true;

			DialogResult dialogResult = MessageBox.Show(this, "Do you want to save changes to the tracklist?", "Warning", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Warning);
			if (dialogResult == DialogResult.Yes)
			{
				if (Properties.Settings.Default.TrackListFileName.Length != 0)
				{
					SaveTracklist(Properties.Settings.Default.TrackListFileName);
				}
				else
				{
					if (!ShowSaveTracklistDialog())
						return false;
				}
			}
			else if (dialogResult != DialogResult.No)
			{
				return false;
			}

			return true;
		}

		bool ShowOpenTracklistDialog()
		{
			DialogResult dialogResult = openTracklistFileDialog.ShowDialog(this);
			if (dialogResult != DialogResult.OK)
				return false;

			OpenTracklist(openTracklistFileDialog.FileName);
			return true;
		}

		void OpenTracklist(string filename)
		{
			lastSavedTracklist = TracklistFileHandler.Open(filename);
			
			mapListBox.Items.Clear();
			foreach (string map in lastSavedTracklist.Maps)
			{
				mapListBox.Items.Add(map);
			}

			UpdateStartIndexMaximum();
		}

		bool ShowSaveTracklistDialog()
		{
			DialogResult dialogResult = saveTracklistFileDialog.ShowDialog(this);
			if (dialogResult != DialogResult.OK)
				return false;

			return SaveTracklist(saveTracklistFileDialog.FileName);
		}

		bool SaveTracklist(string filename)
		{
			Tracklist currentTracklist = GetCurrentTracklist();
			bool success = TracklistFileHandler.Save(filename, currentTracklist);
			if (success)
			{
				lastSavedTracklist = currentTracklist;

				Properties.Settings.Default.TrackListFileName = filename;
				Properties.Settings.Default.Save();
			}

			return success;
		}

		void UpdateStartIndexMaximum()
		{
			startIndexUpDown.Maximum = mapListBox.Items.Count;
		}

		#region Map Events
		private void addMapButton_Click(object sender, EventArgs e)
		{
			ShowAddMapDialog();
		}

		private void addMenuStripItem_Click(object sender, EventArgs e)
		{
			ShowAddMapDialog();
		}

		private void removeMapButton_Click(object sender, EventArgs e)
		{
			RemoveSelectedMaps();
		}

		private void removeMenuStripItem_Click(object sender, EventArgs e)
		{
			RemoveSelectedMaps();
		}

		private void shuffleMenuStripItem_Click(object sender, EventArgs e)
		{
			Random random = new Random();

			for (int i = mapListBox.Items.Count - 1; i >= 1; --i)
			{
				int randomIndex = random.Next(i);
				object temp = mapListBox.Items[i];
				mapListBox.Items[i] = mapListBox.Items[randomIndex];
				mapListBox.Items[randomIndex] = temp;
			}

			mapListBox.SelectedIndices.Clear();
		}

		private void clearMenuStripItem_Click(object sender, EventArgs e)
		{
			if (mapListBox.Items.Count == 0)
				return;

			if (MessageBox.Show(this, "Clear map list?", "Warning", MessageBoxButtons.YesNo, MessageBoxIcon.Warning) == DialogResult.Yes)
			{
				mapListBox.Items.Clear();
				UpdateStartIndexMaximum();
			}
		}

		private void sortAlphabeticallyAscToolStripMenuItem_Click(object sender, EventArgs e)
		{
			object[] itemsCopy = new object[mapListBox.Items.Count];
			mapListBox.Items.CopyTo(itemsCopy, 0);
			IOrderedEnumerable<object> sortedItemsCopy = itemsCopy.OrderBy(obj => obj.ToString());

			mapListBox.Items.Clear();
			mapListBox.Items.AddRange(sortedItemsCopy.ToArray());

			mapListBox.SelectedIndices.Clear();
		}

		private void sortAlphabeticallyDescToolStripMenuItem_Click(object sender, EventArgs e)
		{
			object[] itemsCopy = new object[mapListBox.Items.Count];
			mapListBox.Items.CopyTo(itemsCopy, 0);
			IOrderedEnumerable<object> sortedItemsCopy = itemsCopy.OrderByDescending(obj => obj.ToString());

			mapListBox.Items.Clear();
			mapListBox.Items.AddRange(sortedItemsCopy.ToArray());

			mapListBox.SelectedIndices.Clear();
		}

		private void dateModifiedAscendingToolStripMenuItem_Click(object sender, EventArgs e)
		{
			string[] itemsCopy = mapListBox.Items.Cast<string>().ToArray();
			var orderedItems = itemsCopy.OrderBy(s => File.GetLastWriteTime(Path.Combine(MapsRootPath, s)));
			mapListBox.Items.Clear();
			mapListBox.Items.AddRange(orderedItems.ToArray());
		}

		private void dateModifiedDescendingToolStripMenuItem_Click(object sender, EventArgs e)
		{
			string[] itemsCopy = mapListBox.Items.Cast<string>().ToArray();
			var orderedItems = itemsCopy.OrderByDescending(s => File.GetLastWriteTime(Path.Combine(MapsRootPath, s)));
			mapListBox.Items.Clear();
			mapListBox.Items.AddRange(orderedItems.ToArray());
		}

		private void moveMapUpButton_Click(object sender, EventArgs e)
		{
			int[] selectedIndicesCopy = new int[mapListBox.SelectedIndices.Count];
			mapListBox.SelectedIndices.CopyTo(selectedIndicesCopy, 0);

			for (int i = 0; i < selectedIndicesCopy.Length; ++i)
			{
				int oldIndex = selectedIndicesCopy[i];
				if (oldIndex > 0)
				{
					int newIndex = oldIndex - 1;
					object temp = mapListBox.Items[newIndex];
					mapListBox.Items[newIndex] = mapListBox.Items[oldIndex];
					mapListBox.Items[oldIndex] = temp;

					selectedIndicesCopy[i] = newIndex;
				}
				else
				{
					int newIndex = mapListBox.Items.Count - 1;
					object temp = mapListBox.Items[oldIndex];
					mapListBox.Items.RemoveAt(oldIndex);
					mapListBox.Items.Add(temp);

					selectedIndicesCopy[i] = newIndex;
				}
			}

			mapListBox.SelectedIndices.Clear();
			foreach (int index in selectedIndicesCopy)
			{
				mapListBox.SelectedIndices.Add(index);
			}
		}

		private void moveMapDownButton_Click(object sender, EventArgs e)
		{
			int[] selectedIndicesCopy = new int[mapListBox.SelectedIndices.Count];
			mapListBox.SelectedIndices.CopyTo(selectedIndicesCopy, 0);

			for (int i = selectedIndicesCopy.Length - 1; i >= 0; --i)
			{
				int oldIndex = selectedIndicesCopy[i];
				if (oldIndex < mapListBox.Items.Count - 1)
				{
					int newIndex = oldIndex + 1;
					object temp = mapListBox.Items[newIndex];
					mapListBox.Items[newIndex] = mapListBox.Items[oldIndex];
					mapListBox.Items[oldIndex] = temp;

					selectedIndicesCopy[i] = newIndex;
				}
				else
				{
					object temp = mapListBox.Items[oldIndex];
					mapListBox.Items.RemoveAt(oldIndex);
					mapListBox.Items.Insert(0, temp);

					selectedIndicesCopy[i] = 0;
				}
			}

			mapListBox.SelectedIndices.Clear();
			foreach (int index in selectedIndicesCopy)
			{
				mapListBox.SelectedIndices.Add(index);
			}
		}

		private void addFolderToolStripMenuItem_Click(object sender, EventArgs e)
		{
			RandomMapSelector randomMapSelector = new RandomMapSelector(MapsRootPath);
			if (randomMapSelector.ShowDialog(this) == DialogResult.OK)
			{
				foreach (string map in randomMapSelector.SelectedMaps)
				{
					AddMap(map);
				}
			}
		}
		#endregion

		void ShowAddMapDialog()
		{
			DialogResult dialogResult = addMapFileDialog.ShowDialog(this);
			if (dialogResult != DialogResult.OK)
				return;

			foreach (string file in addMapFileDialog.FileNames)
			{
				AddMap(file);
			}
		}

		void AddMap(string filename)
		{
			string relativeFile = Utilities.GetRelativePath(filename, MapsRootPath);
			if (!mapListBox.Items.Contains(relativeFile))
			{
				mapListBox.Items.Add(relativeFile);
				UpdateStartIndexMaximum();
			}
		}

		void RemoveSelectedMaps()
		{
			object[] selectedItemsCopy = new object[mapListBox.SelectedItems.Count];
			mapListBox.SelectedItems.CopyTo(selectedItemsCopy, 0);
			foreach (object item in selectedItemsCopy)
			{
				mapListBox.Items.Remove(item);
				UpdateStartIndexMaximum();
			}
		}

		#region Options Events
		private void serverExecutableButton_Click(object sender, EventArgs e)
		{
			ShowLocateServerExecutableDialog();
		}

		private void configFileButton_Click(object sender, EventArgs e)
		{
			DialogResult dialogResult = locateConfigFileDialog.ShowDialog(this);
			if (dialogResult != DialogResult.OK)
				return;

			Properties.Settings.Default.ConfigFileName = locateConfigFileDialog.FileName;
			Properties.Settings.Default.Save();
		}
		#endregion

		bool ShowLocateServerExecutableDialog()
		{
			DialogResult dialogResult = locateServerExecutableDialog.ShowDialog(this);
			if (dialogResult == DialogResult.OK)
			{
				Properties.Settings.Default.ServerExecutableFileName = locateServerExecutableDialog.FileName;
				Properties.Settings.Default.Save();

				OnServerExecutableFileNameChanged();
				return true;
			}

			return false;
		}

		void OnServerExecutableFileNameChanged()
		{
			string serverExecutableDirectory = Path.GetDirectoryName(Properties.Settings.Default.ServerExecutableFileName);

			MapsRootPath = Path.Combine(serverExecutableDirectory, "UserData\\Maps\\");
			addMapFileDialog.InitialDirectory = MapsRootPath;

			locateConfigFileDialog.InitialDirectory = Path.Combine(serverExecutableDirectory, "UserData\\Config\\");
			openTracklistFileDialog.InitialDirectory = Path.Combine(serverExecutableDirectory, "UserData\\Maps\\MatchSettings\\");
			saveTracklistFileDialog.InitialDirectory = openTracklistFileDialog.InitialDirectory;

			mapsPanel.Enabled = true;
			configFileButton.Enabled = true;
			startServerButton.Enabled = true;
			copyMapsFromDocumentsButton.Enabled = true;
			mainMenuStrip.Enabled = true;
		}

		private void startServerButton_Click(object sender, EventArgs e)
		{
			if (Properties.Settings.Default.ServerExecutableFileName.Length == 0)
			{
				MessageBox.Show(this, "Server executable path not set.");
				return;
			}

			if (Properties.Settings.Default.ConfigFileName.Length == 0)
			{
				MessageBox.Show(this, "Config file path not set.");
				return;
			}

			if (Properties.Settings.Default.TrackListFileName.Length == 0)
			{
				if (!ShowSaveTracklistDialog())
					return;
			}

			if (IsTracklistDirty())
			{
				DialogResult dialogResult = MessageBox.Show(this, "Do you want to save changes to the tracklist?", "Warning", MessageBoxButtons.YesNo, MessageBoxIcon.Warning);
				if (dialogResult == DialogResult.Yes)
				{
					if (Properties.Settings.Default.TrackListFileName.Length != 0)
					{
						SaveTracklist(Properties.Settings.Default.TrackListFileName);
					}
					else
					{
						if (!ShowSaveTracklistDialog())
							return;
					}
				}
				else
				{
					return;
				}
			}

			ProcessStartInfo processStartInfo = new ProcessStartInfo();
			processStartInfo.UseShellExecute = true;

			string serverExecutableDirectory = Path.GetDirectoryName(Properties.Settings.Default.ServerExecutableFileName);
			processStartInfo.WorkingDirectory = serverExecutableDirectory;
			processStartInfo.FileName = Path.GetFileName(Properties.Settings.Default.ServerExecutableFileName);

			string configRelativePath = Utilities.GetRelativePath(Properties.Settings.Default.ConfigFileName, serverExecutableDirectory + "\\UserData\\Config\\");
			string tracklistRelativePath = Utilities.GetRelativePath(Properties.Settings.Default.TrackListFileName, serverExecutableDirectory + "\\UserData\\Maps\\");
			processStartInfo.Arguments = "/title=Trackmania /game_Settings=" + tracklistRelativePath + " /dedicated_cfg=" + configRelativePath;

			Enabled = false;

			try
			{
				using (Process serverProcess = Process.Start(processStartInfo))
				{
					Settings.Default.LastServerBootTime = DateTime.Now;
					Settings.Default.Save();

					serverProcess.WaitForExit();
				}
			}
			catch (Exception ex)
			{
				MessageBox.Show(this, "Could not start server process: " + ex.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
			}

			Enabled = true;
		}

		private void copyMapsFromDocumentsButton_Click(object sender, EventArgs e)
		{
			CopyFromMyMaps copyFromMyMaps = new CopyFromMyMaps(MapsRootPath);
			copyFromMyMaps.ShowDialog(this);
		}

		
	}
}
