﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using TrackmaniaServerTool.Properties;

namespace TrackmaniaServerTool
{
	public partial class RandomMapSelector : Form
	{
		public RandomMapSelector(string mapsRootFolder)
		{
			InitializeComponent();

			folderTextBox.Text = SelectedFolder = MapsRootFolder = mapsRootFolder;
			SelectedMaps = new List<string>();

			mapSelectionMethodComboBox.SelectedIndex = 0;

			fromDatePicker.Value = Settings.Default.LastServerBootTime;
			toDatePicker.Value = DateTime.Now;
			toDatePicker.ValueChanged += toDatePicker_ValueChanged;

			fromDatePicker.MaxDate = toDatePicker.Value;
			toDatePicker.MinDate = fromDatePicker.Value;
		}

		string MapsRootFolder;
		string SelectedFolder;

		public List<string> SelectedMaps { get; private set; }

		public enum MapSelectionMethods
		{
			Random,
			DateRange,
			SearchTerm,
			All,
		}
		public MapSelectionMethods MapSelectionMethod { get; private set; }

		private void folderButton_Click(object sender, EventArgs e)
		{
			FolderBrowserDialog2 folderBrowserDialog2 = new FolderBrowserDialog2();
			folderBrowserDialog2.DirectoryPath = MapsRootFolder;

			DialogResult dialogResult = folderBrowserDialog2.ShowDialog(this);
			if (dialogResult != DialogResult.OK)
				return;

			folderTextBox.Text = SelectedFolder = folderBrowserDialog2.DirectoryPath;
		}

		private void toDatePicker_ValueChanged(object sender, EventArgs e)
		{
			toDatePicker.Value = new DateTime(toDatePicker.Value.Year, toDatePicker.Value.Month, toDatePicker.Value.Day, 23, 59, 59, 999, DateTimeKind.Local);
			fromDatePicker.MaxDate = toDatePicker.Value;
		}

		private void fromDatePicker_ValueChanged(object sender, EventArgs e)
		{
			fromDatePicker.Value = new DateTime(fromDatePicker.Value.Year, fromDatePicker.Value.Month, fromDatePicker.Value.Day, 0, 0, 0, 0, DateTimeKind.Local);
			toDatePicker.MinDate = fromDatePicker.Value;
		}

		private void okButton_Click(object sender, EventArgs e)
		{
			switch (MapSelectionMethod)
			{
			case MapSelectionMethods.Random:
				SelectRandomMaps();
				break;

			case MapSelectionMethods.DateRange:
				SelectDateRangeMaps();
				break;

			case MapSelectionMethods.SearchTerm:
				SelectSearchTermMaps();
				break;

			case MapSelectionMethods.All:
				SelectedMaps = Directory.GetFiles(SelectedFolder, "*.Map.Gbx", SearchOption.AllDirectories).ToList();
				break;
			}

			if (SelectedMaps.Count > 50)
			{
				if (MessageBox.Show(this, "About to add " + SelectedMaps.Count + " maps, continue?", "Warning", MessageBoxButtons.YesNo, MessageBoxIcon.Warning) != DialogResult.Yes)
				{
					SelectedMaps.Clear();
					return;
				}
			}

			DialogResult = DialogResult.OK;
			Close();
		}

		private void cancelButton_Click(object sender, EventArgs e)
		{
			DialogResult = DialogResult.Cancel;
			Close();
		}

		private void mapSelectionMethodComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			MapSelectionMethod = (MapSelectionMethods)mapSelectionMethodComboBox.SelectedIndex;

			switch (MapSelectionMethod)
			{
			case MapSelectionMethods.Random:
				numberOfMapsPanel.Visible = true;
				datePickerPanel.Visible = false;
				searchTermPanel.Visible = false;
				break;

			case MapSelectionMethods.DateRange:
				numberOfMapsPanel.Visible = false;
				datePickerPanel.Visible = true;
				searchTermPanel.Visible = false;
				break;

			case MapSelectionMethods.SearchTerm:
				numberOfMapsPanel.Visible = false;
				datePickerPanel.Visible = false;
				searchTermPanel.Visible = true;
				break;

			case MapSelectionMethods.All:
				numberOfMapsPanel.Visible = false;
				datePickerPanel.Visible = false;
				searchTermPanel.Visible = false;
				break;
			}
		}

		void SelectRandomMaps()
		{
			string[] allMaps = Directory.GetFiles(SelectedFolder, "*.Map.Gbx", SearchOption.AllDirectories);

			Random random = new Random();
			for (int i = allMaps.Length - 1; i >= 1; --i)
			{
				int randomIndex = random.Next(i);
				string temp = allMaps[i];
				allMaps[i] = allMaps[randomIndex];
				allMaps[randomIndex] = temp;
			}

			SelectedMaps = allMaps.ToList();

			int numberOfMapsToChoose = Math.Min((int)numberOfMapsUpDown.Value, allMaps.Length);
			SelectedMaps.RemoveRange(numberOfMapsToChoose, allMaps.Length - numberOfMapsToChoose);
		}

		void SelectDateRangeMaps()
		{
			foreach (string map in Directory.EnumerateFiles(SelectedFolder, "*.Map.Gbx", SearchOption.AllDirectories))
			{
				DateTime creationTime = File.GetLastWriteTime(map);
				if ((fromDatePicker.Value == null || creationTime >= fromDatePicker.Value) &&
					(toDatePicker.Value == null || creationTime <= toDatePicker.Value))
				{
					SelectedMaps.Add(map);
				}
			}
		}

		void SelectSearchTermMaps()
		{
			string searchTerm = searchTermTextBox.Text.Trim();
			if (searchTerm.Length == 0)
				return;

			foreach (string map in Directory.EnumerateFiles(SelectedFolder, "*.Map.Gbx", SearchOption.AllDirectories))
			{
				string fileName = Path.GetFileNameWithoutExtension(map);
				if (fileName.EndsWith(".Map"))
				{
					fileName = fileName.Remove(fileName.Length - 4, 4);
				}

				if (CultureInfo.CurrentCulture.CompareInfo.IndexOf(fileName, searchTermTextBox.Text, CompareOptions.IgnoreCase) >= 0)
				{
					SelectedMaps.Add(map);
				}
			}
		}
	}
}
